#!/usr/bin/python3

"""
Basic unit tests for rename-flac
"""

import os
import shutil

from types import SimpleNamespace

import pytest

from rename_flac import scheme_processing, fetch_metadata, rename

def test_tracknumber_title():
    """Test scheme function with tracknumber and title"""
    args = SimpleNamespace(scheme='%n - %t')
    scheme = scheme_processing(args)
    assert scheme == '%(tracknumber)s - %(title)s'

def test_tracknumber_title_extra():
    """Test scheme function with tracknumber, title and some extra text"""
    args = SimpleNamespace(scheme='%n - %t Foobars200')
    scheme = scheme_processing(args)
    assert scheme == '%(tracknumber)s - %(title)s Foobars200'

def test_all():
    """Test scheme function with all the options"""
    args = SimpleNamespace(scheme='%a - %b - %c - %d - %g - %n - %t')
    scheme = scheme_processing(args)
    assert scheme == ("%(artist)s - %(album)s - %(composer)s - "
                     "%(date)s - %(genre)s - %(tracknumber)s - %(title)s")

def test_no_tracknumber_or_title():
    """Test scheme function failing when there's neither tracknumber nor title"""
    args = SimpleNamespace(scheme='%a')
    with pytest.raises(ValueError):
        assert scheme_processing(args)

def test_metadata():
    """Test the fetch_metadata function"""
    metadata = fetch_metadata('tests/blank.flac')
    assert metadata['artist'] == 'Artist'
    assert metadata['album'] == 'Album'
    assert metadata['composer'] == 'Composer'
    assert metadata['date'] == 'Date'
    assert metadata['genre'] == 'Genre'
    assert metadata['tracknumber'] == 'Tracknumber'
    assert metadata['title'] == 'Title'

@pytest.fixture
def outdir(tmp_path):
    """Create temporary dir with the testfile'"""
    out_dir = tmp_path / "out"
    out_dir.mkdir()
    shutil.copy('tests/blank-with-slash.flac',
                os.path.join(out_dir, 'blank-with-slash.flac'))

    return out_dir

def test_sanitize_slash(outdir):  # pylint: disable=W0621
    """Test --sanitize_slash argument"""
    args = SimpleNamespace(sanitize_slash='∕', scheme='%n - %t', verbose=True)
    scheme = scheme_processing(args)
    rename(scheme, outdir, 'blank-with-slash.flac', args)
    assert os.path.exists(os.path.join(outdir,
                                       'Tracknumber - Title ∕ Title 2.flac'))
